
/*******************************************************************************/
/*  
	This script cleans the PSID data and creates hand-to-mouth measures following KVW 2014. 
	Download PSID Main Study 2017 here: https://simba.isr.umich.edu/Zips/ZipMain.aspx
    --
	Alexander Weinberg
	February 16, 2021
*/
/*******************************************************************************/

clear all
set maxvar 30000

//________________________________________________________________
// Import PSID 
// Cleaning following Kaplan, Violante, and Weidner (Brookings 2014)
// Some code is directly from their replication package.

do "../Raw Data/PSID/fam2017er/FAM2017ER.do"
#delimit cr

//________________________________________________________________
// Variable selection
keep ER66002 ER66009 ER66017 ER71331 ER71333 ER71335 ER66018 ///
	ER71347 ER71349 ER71351 ER71353 ER71305 ER71377 ER71363 ER71361 ///
	ER66030  ER70908 ER66030 ER66172 ER71305 ER71277 ER70874 ER70882 ///
	ER66170 ER71365  ER66024 ///
	ER71277 ER71427 ER71429 ER71433 ER71435 ER71437 ER71439 ER71443 	///
	ER71445 ER71447 ER71449 ER71451 ER71453 ER71455 ER71457 ER71459	///
	ER71461 ER71463 ER71465 ER71467 ER71469 ER71471 ER71473 ER71475 ER71481 ///
	ER71483 ER71485 ER71570 ///
		ER71512 ER66195 ER66196 ER66197 
		
//________________________________________________________________
// Zeros out dont know and NA codes
replace ER71305 = 0 if ER71305 >= 9999998
local varszero "ER71347 ER71377 ER71351 ER71349 ER71353 ER71333 ER71363 ER71331 ER71361 ER71335 ER71365"
foreach k of local varszero {
replace `k' = 0 if `k' >= 999998
}

//________________________________________________________________
// Labor market income = wages and salaries + UIben + workers comp + child support
// + alimony + SSI + ADC/AFDC + other welfare
gen labinc = (ER71305 + ER71277) + (ER71347 + ER71377) + ER71351 + ///
ER71349 + ER71353 + (ER71333 + ER71363) + (ER71331 + ER71361) + ///
(ER71335 + ER71365)

gen year = 2017
rename ER66017 age
replace age = 0 if age == 999
rename ER66009 famnum
rename ER66002 hhid
rename ER71570 wgt
rename ER7151 cons_trans
rename ER66195 occ
rename ER66196 ind2012
rename ER66197 commute

//________________________________________________________________
// Renames wealth variables
// Net value of farm and business
rename ER71427 hasbus
rename ER71429 bus
// Value of checking and savings accounts, money market funds, certificates of 
// deposit, savings bonds, Treasury bills
rename ER71433 haschecking
rename ER71435 checking
// Net value of real estate other than main home
rename ER71437 hasothrealestate
rename ER71439 othrealestate
// Value of shares of stock in publicly held corporations, mutual funds or investment trusts
rename ER71443 hasstocks
rename ER71445 stocks
// Net value of vehicles or other assets 'on wheels'
rename ER71447 vehic
// Value of other investments in trusts or estates, bond funds, life insurance //
// policies, special collections
rename ER71449 hasothassets
rename ER71451 othassets
// Value of private annuities or IRAs
rename ER71453 hasira
rename ER71455 ira
// Value of home equity
rename ER71481 homeequity
// credit card or store care debt
rename ER71457 hasccdebt
rename ER71459 ccdebt
// other debts such as student loans
rename ER71461 hasstudentdebt
rename ER71463 studentdebt
// medical bills
rename ER71465 hasmedicaldebt
rename ER71467 medicaldebt
// legal bills
rename ER71469 haslegaldebt
rename ER71471 legaldebt
// family loans
rename ER71473 hasfamdebt
rename ER71475 famdebt

rename ER71483 networthnohomeequity
rename ER71485 networth

//________________________________________________________________
// Generates wealth variables like the SCF code
gen liqpos      = checking
gen liqneg      = ccdebt
gen direct      = stocks
gen nethouses   = homeequity + othrealestate
gen netcars     = vehic
gen retacc      = ira
gen lifeins     = othassets
gen netbus      = bus
gen othdebt     = studentdebt + medicaldebt + legaldebt + famdebt
gen brliqpos      = liqpos + direct
gen brliqneg      = liqneg
gen netbrliq      = brliqpos - brliqneg
gen netbrilliq    = nethouses + netcars + retacc + lifeins + 0*netbus
gen netbrilliqnc  = netbrilliq - netcars
gen networthnc  = netbrilliqnc + netbrliq
gen netbrilliqbusnc = netbrilliqnc + netbus
gen netbrliqnstocks = netbrliq - direct
gen netbrilliqncstocks = netbrilliqnc + direct
gen networthbusnc  = netbrilliqbusnc + netbrliq
gen networthcars = netbrilliqnc + netbrliq + netcars

//________________________________________________________________
// Labor income 
rename hhid intid
gen labinc_post = labinc
drop labinc

//________________________________________________________________
// Relevant variables for all of the htm calcs
gen monthlabinc = labinc_post/12
gen m1 		 	= 1*monthlabinc
gen m6 		 	= 6*monthlabinc
gen m12 	 	= 12*monthlabinc
gen liqvar   	= netbrliq          //netbrliq brliqpos
gen illiqvar 	= netbrilliqnc      //netbrilliqnc netbrilliq
gen nwvar    	= networthnc        //networthnew networthnc

//________________________________________________________________
// Hand-to-Mouth Calculation
local payper "24"			// Bi-weekly pay period
foreach p of local payper{

	//________________________________________________________________
	// No borrowing
	replace liqvar 		= netbrliq          //netbrliq brliqpos
	replace illiqvar 	= netbrilliqnc          //netbrilliqnc netbrilliq
	replace nwvar    = networthnc        //networthnew networthnc

	gen htm0_a_`p'_0x = liqvar<=labinc_post/(2*`p')
	gen htm0_b_`p'_0x = liqvar<=labinc_post/(2*`p') & illiqvar>0
	gen htm0_c_`p'_0x = nwvar<=labinc_post/(2*`p')

	gen h2m_`p'_0x = (htm0_a_`p'_0x == 1)
	gen Wh2m_`p'_0x = (htm0_b_`p'_0x == 1)
	gen Ph2m_`p'_0x = (htm0_a_`p'_0x == 1 & htm0_b_`p'_0x == 0)
	gen Nh2m_`p'_0x = (htm0_a_`p'_0x == 0)
	gen h2mNW_`p'_0x = (htm0_c_`p'_0x == 1)

	gen h2m_status_`p'_0x = 1 if Wh2m_`p'_0x == 1
	replace h2m_status_`p'_0x = 2 if Ph2m_`p'_0x == 1
	replace h2m_status_`p'_0x = 3 if Nh2m_`p'_0x == 1

	//________________________________________________________________
	// Access to 12x monthly income of credit
	replace liqvar = netbrliq          //netbrliq brliqpos
	replace illiqvar = netbrilliqnc          //netbrilliqnc netbrilliq
	replace nwvar    = networthnc        //networthnew networthnc

	gen htm0_a_`p'_12x = (liqvar>=0 & liqvar<=labinc_post/(2*`p')) | (liqvar<0 & liqvar<=(labinc_post/`p')/2 - m12)
	gen htm0_b_`p'_12x = (liqvar>=0 & liqvar<=labinc_post/(2*`p') & illiqvar>0) | (liqvar<0 & liqvar<=(labinc_post/`p')/2 - m12 & illiqvar>0)
	gen htm0_c_`p'_12x = (nwvar>=0 & nwvar<=labinc_post/(2*`p')) | (nwvar<0 & nwvar<=(labinc_post/`p')/2 - m12)

	gen h2m_`p'_12x = (htm0_a_`p'_12x == 1)
	gen Wh2m_`p'_12x = (htm0_b_`p'_12x == 1)
	gen Ph2m_`p'_12x = (htm0_a_`p'_12x == 1 & htm0_b_`p'_12x == 0)
	gen Nh2m_`p'_12x = (htm0_a_`p'_12x == 0)
	gen h2mNW_`p'_12x = (htm0_c_`p'_12x == 1)

	gen h2m_status_`p'_12x = 1 if Wh2m_`p'_12x == 1
	replace h2m_status_`p'_12x = 2 if Ph2m_`p'_12x == 1
	replace h2m_status_`p'_12x = 3 if Nh2m_`p'_12x == 1

	//________________________________________________________________
	// access to 6x monthly income of credit
	replace liqvar = netbrliq          		 //netbrliq brliqpos
	replace illiqvar = netbrilliqnc          //netbrilliqnc netbrilliq
	replace nwvar    = networthnc        	 //networthnew networthnc

	gen htm0_a_`p'_6x = (liqvar>=0 & liqvar<=labinc_post/(2*`p')) | (liqvar<0 & liqvar<=(labinc_post/`p')/2 - m6)
	gen htm0_b_`p'_6x = (liqvar>=0 & liqvar<=labinc_post/(2*`p') & illiqvar>0) | (liqvar<0 & liqvar<=(labinc_post/`p')/2 - m6 & illiqvar>0)
	gen htm0_c_`p'_6x = (nwvar>=0 & nwvar<=labinc_post/(2*`p')) | (nwvar<0 & nwvar<=(labinc_post/`p')/2 - m6)

	gen h2m_`p'_6x 		= (htm0_a_`p'_6x == 1)
	gen Wh2m_`p'_6x 	= (htm0_b_`p'_6x == 1)
	gen Ph2m_`p'_6x 	= (htm0_a_`p'_6x == 1 & htm0_b_`p'_6x == 0)
	gen Nh2m_`p'_6x 	= (htm0_a_`p'_6x == 0)
	gen h2mNW_`p'_6x 	= (htm0_c_`p'_6x == 1)

	gen h2m_status_`p'_6x = 1 if Wh2m_`p'_6x == 1
	replace h2m_status_`p'_6x = 2 if Ph2m_`p'_6x == 1
	replace h2m_status_`p'_6x = 3 if Nh2m_`p'_6x == 1

	//________________________________________________________________
	// access to 1x monthly income of credit
	/* THIS IS THE BENCHMARK */
	replace liqvar 		= netbrliq          // netbrliq brliqpos
	replace illiqvar 	= netbrilliqnc      // netbrilliqnc netbrilliq
	replace nwvar    	= networthnc        // networthnew networthnc

	gen htm0_a_`p'_1x 	= (liqvar>=0 & liqvar<=labinc_post/(2*`p')) | (liqvar<0 & liqvar<=(labinc_post/`p')/2 - m1)
	gen htm0_b_`p'_1x 	= (liqvar>=0 & liqvar<=labinc_post/(2*`p') & illiqvar>0) | (liqvar<0 & liqvar<=(labinc_post/`p')/2 - m1 & illiqvar>0)
	gen htm0_c_`p'_1x 	= (nwvar>=0 & nwvar<=labinc_post/(2*`p')) | (nwvar<0 & nwvar<=(labinc_post/`p')/2 - m1)

	gen h2m_`p'_1x 		= (htm0_a_`p'_1x == 1) 							// Hand to mouth
	gen Wh2m_`p'_1x 	= (htm0_b_`p'_1x == 1) 							// Wealthy HtM
	gen Ph2m_`p'_1x 	= (htm0_a_`p'_1x == 1 & htm0_b_`p'_1x == 0) 	// Poor HtM
	gen Nh2m_`p'_1x 	= (htm0_a_`p'_1x == 0) 							// Not HtM
	gen h2mNW_`p'_1x 	= (htm0_c_`p'_1x == 1) 							// 

	/* h2m_status_24_1x */
	gen 	h2m_status_`p'_1x = 1 if Wh2m_`p'_1x == 1 	// Wealthy
	replace h2m_status_`p'_1x = 2 if Ph2m_`p'_1x == 1 	// Poor
	replace h2m_status_`p'_1x = 3 if Nh2m_`p'_1x == 1 	// Non


}


//________________________________________________________________
// Generate Hand-to-Mouth Measures

// 1x = Access to credit = to one month of salary 
// 24 = 24 pay periods in a year, i.e. bi-weekly
gen htm_baseline = h2m_status_24_1x

//________________________________________________________________
// Generate two indicators
// 1. Hand to mouth
gen h2m = (htm_baseline < 3)
// 2. Wealthy hand to mouth
gen Wh2m = .
replace Wh2m = 0 if (htm_baseline == 2) 	// Poor HtM
replace Wh2m = 1 if (htm_baseline == 1) 	// Wealthy HtM

label def h2m_label 1 "HtM" 0 "Non HtM"
label values h2m h2m_label

label def Wh2m_label 0 "Poor HtM" 1 "Wealthy HtM" 
label values Wh2m Wh2m_label

//________________________________________________________________
// Demographics
gen male 	 		= (ER66018 == 1) //Sex of reference person
gen own_home 		= (ER66030 == 1) //Own/rent

gen college  		= (ER70908 == 1) // college degree
replace college 	= . if ER70908==9

gen famwage 		= (ER71305 + ER71277)

gen uhrswork1 		= ER66172 // weekly horus
replace uhrswork1 	= . if uhrswork1 > 112 | (uhrswork1 == 0)

gen parttime 		= (uhrswork1 < 30)
gen fulltime    	= 1 - parttime

gen married 		= (ER66024==1)

gen born_50states 	= inrange(ER70874, 1, 56) // Born in US
replace born_50states = . if ER70874==99

gen white 			= (ER70882==1) // race
replace white 		= . if ER70882==9

gen no_unemp 		= (ER66170==52) // weeks unemployed last year
replace no_unemp 	= . if ER66170==0 | ER66170>98

gen old 			= (age >=50)

//________________________________________________________________
// LABEL VARS
label var college "College degree"
label var male "Male"
label var own_home "Owns home"
label var born_50states "Born in US"
label var married "Married"
label var white "White"
label var fulltime "Full-time employed"
label var old "Age 50+"
label var networth "Net worth"
label var age "Age"
label var famwage "Family income from wages and salaries"
label var uhrswork1 "Usual weekly hours"
label var labinc "Labor income incl. welfare"
label var h2m "Hand-to-mouth"
label var Wh2m "Wealthy hand-to-mouth"


//________________________________________________________________
// Save data
keep if year==2017
save "../Data/PSID/psid_occ_h2m", replace


//end
